﻿using Regatron.G5.Common;
using System;
using System.Collections.Generic;
using System.Windows.Input;
using G5.Api.Example.Model;
using System.Globalization;

namespace G5.Api.Example.ViewModel
{
  internal class FunctionGeneratorTabControlViewModel : BindableBase, IFunctionGeneratorTabControlViewModel
  {
    private readonly IFunctionGeneratorService _functionGeneratorService;
    private readonly IConnectionService _connectionService;
    private readonly CultureInfo _cultureInfo;

    private FunctionGeneratorControlMode _functionGeneratorControlModeSelection;
    private FunctionGeneratorFunctionType _functionGeneratorFunctionTypeSelection;
    private uint _blockFunctionRepetitions;
    private string _blockFunctionRepetitionsString;
    private float _blockPeakValue;
    private string _blockPeakValueString;
    private float _blockOffset;
    private string _blockOffsetString;
    private float _blockFrequency;
    private string _blockFrequencyString;
    private float _blockSymmetry;
    private string _blockSymmetryString;

    private FunctionGeneratorTriggerMode _functionGeneratorTriggerModeSelection;
    private FunctionGeneratorSequenceFinishBehaviour _functionGeneratorSequenceFinishBehaviourSelection;
    private uint _blockRepetitions;
    private string _blockRepetitionsString;
    private float _sequenceDelay;
    private string _sequenceDelaystring;

    private ICommand _functionGeneratorEnableCommand;
    private ICommand _functionGeneratorDisableCommand;
    private ICommand _functionGeneratorStartCommand;
    private ICommand _functionGeneratorStopCommand;
    private bool _functionGeneratorIsRunning;
    private bool _functionGeneratorIsEnabled;

    private float _mppVoltage;
    private string _mppVoltageString;
    private float _mppCurrent;
    private string _mppCurrentString;
    private SasCellTechnology _solarArrayCellTechnologySelection;
    private string _actualIrradianceString;
    private float _actualIrradiance;
    private string _actualTemperatureString;
    private float _actualTemperature;

    public List<FunctionGeneratorControlMode> FunctionGeneratorControlModes { get; private set; }
    public List<FunctionGeneratorTriggerMode> FunctionGeneratorTriggerModes { get; private set; }
    public List<FunctionGeneratorSequenceFinishBehaviour> FunctionGeneratorSequenceFinishBehaviors { get; private set; }
    public List<FunctionGeneratorFunctionType> FunctionGeneratorFunctionTypes { get; private set; }
    public List<SasCellTechnology> SolarArrayCellTechnologies { get; private set; }
    public ICellTechnologyParametersViewModel CellTechnologyParametersViewModel { get; }

    public FunctionGeneratorControlMode FunctionGeneratorControlModeSelection
    {
      get => _functionGeneratorControlModeSelection;
      set
      {
        if (SetProperty(ref _functionGeneratorControlModeSelection, value))
        {
          _functionGeneratorService.SetControlMode(value);
          UpdateFunctionGeneratorValuesFromDevice();
        }
      }
    }
       
    public FunctionGeneratorFunctionType FunctionGeneratorFunctionTypeSelection
    {
      get => _functionGeneratorFunctionTypeSelection;
      set
      {
        if (SetProperty(ref _functionGeneratorFunctionTypeSelection, value))
        {
          _functionGeneratorService.SetBaseFunction(value);
          UpdateFunctionGeneratorValuesFromDevice();
        }
      }
    }

    public string BlockFunctionRepetitionsString
    {
      get => _blockFunctionRepetitionsString;
      set
      {
        if (uint.TryParse(value, out _blockFunctionRepetitions))
        {
          if (SetProperty(ref _blockFunctionRepetitionsString, value))
          {
            _functionGeneratorService.SetBlockFunctionRepetitions(_blockFunctionRepetitions);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _blockFunctionRepetitionsString, String.Empty);
        }
      }
    }

    public string BlockPeakValueString
    {
      get => _blockPeakValueString;
      set
      {
        if (float.TryParse(value, out _blockPeakValue))
        {
          if (SetProperty(ref _blockPeakValueString, value))
          {
            _functionGeneratorService.SetPeakValue(_blockPeakValue);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _blockPeakValueString, String.Empty);
        }
      }
    }

    public string BlockOffsetString
    {
      get => _blockOffsetString;
      set
      {
        if (float.TryParse(value, out _blockOffset))
        {
          if (SetProperty(ref _blockOffsetString, value))
          {
            _functionGeneratorService.SetOffset(_blockOffset);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _blockOffsetString, String.Empty);
        }
      }
    }

    public string BlockFrequencyString
    {
      get => _blockFrequencyString;
      set
      {
        if (float.TryParse(value, out _blockFrequency))
        {
          if (SetProperty(ref _blockFrequencyString, value))
          {
            _functionGeneratorService.SetFrequency(_blockFrequency);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _blockFrequencyString, String.Empty);
        }
      }
    }

    public string BlockSymmetryString
    {
      get => _blockSymmetryString;
      set
      {
        if (float.TryParse(value, out _blockSymmetry))
        {
          if (SetProperty(ref _blockSymmetryString, value))
          {
            _functionGeneratorService.SetSymmetry(_blockSymmetry);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _blockSymmetryString, String.Empty);
        }
      }
    }

    public FunctionGeneratorTriggerMode FunctionGeneratorTriggerModeSelection
    {
      get => _functionGeneratorTriggerModeSelection;
      set
      {
        if (SetProperty(ref _functionGeneratorTriggerModeSelection, value))
        {
          _functionGeneratorService.SetTriggerMode(_functionGeneratorTriggerModeSelection);
          UpdateFunctionGeneratorValuesFromDevice();
        }
      }
    }

    public FunctionGeneratorSequenceFinishBehaviour FunctionGeneratorSequenceFinishBehaviourSelection
    {
      get => _functionGeneratorSequenceFinishBehaviourSelection;
      set
      {
        if (SetProperty(ref _functionGeneratorSequenceFinishBehaviourSelection, value))
        {
          _functionGeneratorService.SetSequenceFinishBehavior(_functionGeneratorSequenceFinishBehaviourSelection);
          UpdateFunctionGeneratorValuesFromDevice();
        }
      }
    }

    public string BlockRepetitionsString
    {
      get => _blockRepetitionsString;
      set
      {
        if (uint.TryParse(value, out _blockRepetitions))
        {
          if (SetProperty(ref _blockRepetitionsString, value))
          {
            _functionGeneratorService.SetBlockRepetitions(_blockRepetitions);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _blockRepetitionsString, String.Empty);
        }
      }
    }

    public string SequenceDelayString
    {
      get => _sequenceDelaystring;
      set
      {
        if (float.TryParse(value, out _sequenceDelay))
        {
          if (SetProperty(ref _sequenceDelaystring, value))
          {
            _functionGeneratorService.SetSequenceDelay(_sequenceDelay);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _sequenceDelaystring, String.Empty);
        }
      }
    }

    public ICommand FunctionGeneratorEnableCommand
    {
      get => _functionGeneratorEnableCommand;
      set => SetProperty(ref _functionGeneratorEnableCommand, value);
    }

    public ICommand FunctionGeneratorDisableCommand
    {
      get => _functionGeneratorDisableCommand;
      set => SetProperty(ref _functionGeneratorDisableCommand, value);
    }

    public ICommand FunctionGeneratorStartCommand
    {
      get => _functionGeneratorStartCommand;
      set => SetProperty(ref _functionGeneratorStartCommand, value);
    }

    public ICommand FunctionGeneratorStopCommand
    {
      get => _functionGeneratorStopCommand;
      set => SetProperty(ref _functionGeneratorStopCommand, value);
    }

    public bool FunctionGeneratorIsRunning
    {
      get => _functionGeneratorIsRunning;
      set => SetProperty(ref _functionGeneratorIsRunning, value);
    }

    public bool FunctionGeneratorIsEnabled
    {
      get => _functionGeneratorIsEnabled;
      set => SetProperty(ref _functionGeneratorIsEnabled, value);
    }
    public ICommand UpdateValuesFromDeviceCommand { get; }

    public string MppVoltageString
    {
      get => _mppVoltageString;
      set
      {
        if (float.TryParse(value, out _mppVoltage))
        {
          SetProperty(ref _mppVoltageString, value);
        }
        else
        {
          SetProperty(ref _mppVoltageString, String.Empty);
        }
      }
    }

    public string MppCurrentString
    {
      get => _mppCurrentString;
      set
      {
        if (float.TryParse(value, out _mppCurrent))
        {
          SetProperty(ref _mppCurrentString, value);
        }
        else
        {
          SetProperty(ref _mppCurrentString, String.Empty);
        }
      }
    }

    public SasCellTechnology SolarArrayCellTechnologySelection
    {
      get => _solarArrayCellTechnologySelection;
      set 
      {
        SetProperty(ref _solarArrayCellTechnologySelection, value);
        UpdateCellParameters();
      }
    }

    public string ActualIrradianceString
    {
      get => _actualIrradianceString;
      set
      {
        if (float.TryParse(value, out _actualIrradiance))
        {
          if (SetProperty(ref _actualIrradianceString, value))
          {
            _functionGeneratorService.SetActualIrradiance(_actualIrradiance, 0);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _sequenceDelaystring, String.Empty);
        }
      }
    }

    public string ActualTemperatureString
    {
      get => _actualTemperatureString;
      set
      {
        if (float.TryParse(value, out _actualTemperature))
        {
          if (SetProperty(ref _actualTemperatureString, value))
          {
            _functionGeneratorService.SetActualTemperature(_actualTemperature, 0);
            UpdateFunctionGeneratorValuesFromDevice();
          }
        }
        else
        {
          SetProperty(ref _sequenceDelaystring, String.Empty);
        }
      }
    }

    public ICommand SetSasCurveCommand { get; }

    public FunctionGeneratorTabControlViewModel(IFunctionGeneratorService functionGeneratorService, IConnectionService connectionService, CultureInfo cultureInfo)
    {
      _functionGeneratorService = functionGeneratorService ?? throw new ArgumentNullException();
      _connectionService = connectionService ?? throw new ArgumentNullException();
      _cultureInfo = cultureInfo ?? throw new ArgumentNullException();

      _connectionService.ConnectionChanged += ConnectionChangedHandler;

      FunctionGeneratorEnableCommand = new BaseCommand(notNeededArgument => SetFunctionGeneratorCommand(FunctionGeneratorCommand.Enable), (notNeededArgument) => !FunctionGeneratorIsEnabled);
      FunctionGeneratorDisableCommand = new BaseCommand(notNeededArgument => SetFunctionGeneratorCommand(FunctionGeneratorCommand.Disable), (notNeededArgument) => FunctionGeneratorIsEnabled);
      FunctionGeneratorStartCommand = new BaseCommand(notNeededArgument => SetFunctionGeneratorCommand(FunctionGeneratorCommand.ManualStart), (notNeededArgument) => !FunctionGeneratorIsRunning);
      FunctionGeneratorStopCommand = new BaseCommand(notNeededArgument => SetFunctionGeneratorCommand(FunctionGeneratorCommand.ManualStop), (notNeededArgument) => FunctionGeneratorIsRunning);
      UpdateValuesFromDeviceCommand = new BaseCommand(notNeededArgument => UpdateFunctionGeneratorValuesFromDevice(), (notNeededArgument) => _connectionService.IsConnected);
      SetSasCurveCommand = new BaseCommand(notNeededArgument => SetSasCurve(_mppVoltage, _mppCurrent, SolarArrayCellTechnologySelection), (notNeededArgument) => _connectionService.IsConnected);

      CellTechnologyParametersViewModel = new CellTechnologyParametersViewModel();

      InitComboBoxes();
    }

    private void ConnectionChangedHandler(object sender, ValueChangedEventArgs e)
    {
      UpdateFunctionGeneratorValuesFromDevice();
      UpdateSasSpecificFunctionGeneratorValuesFromDevice();
    }

    private void InitComboBoxes()
    {
      FunctionGeneratorControlModes = new List<FunctionGeneratorControlMode> { FunctionGeneratorControlMode.Voltage, FunctionGeneratorControlMode.Current, FunctionGeneratorControlMode.Power };
      FunctionGeneratorControlModeSelection = FunctionGeneratorControlMode.Voltage;

      FunctionGeneratorFunctionTypes = new List<FunctionGeneratorFunctionType>
      {
        FunctionGeneratorFunctionType.Sine, FunctionGeneratorFunctionType.RectifiedSine, FunctionGeneratorFunctionType.Triangle,
        FunctionGeneratorFunctionType.Rectangle, FunctionGeneratorFunctionType.UserDefined, FunctionGeneratorFunctionType.Aap, FunctionGeneratorFunctionType.SolarArraySimulation
      };
      FunctionGeneratorFunctionTypeSelection = FunctionGeneratorFunctionType.Sine;

      FunctionGeneratorTriggerModes = new List<FunctionGeneratorTriggerMode> { FunctionGeneratorTriggerMode.ExternalInput, FunctionGeneratorTriggerMode.Manual, FunctionGeneratorTriggerMode.VoltageOn };
      FunctionGeneratorTriggerModeSelection = FunctionGeneratorTriggerMode.Manual;

      FunctionGeneratorSequenceFinishBehaviors = new List<FunctionGeneratorSequenceFinishBehaviour>{FunctionGeneratorSequenceFinishBehaviour.HoldLevel,
        FunctionGeneratorSequenceFinishBehaviour.StandardInput, FunctionGeneratorSequenceFinishBehaviour.VoltageOff};
      FunctionGeneratorSequenceFinishBehaviourSelection = FunctionGeneratorSequenceFinishBehaviour.HoldLevel;

      SolarArrayCellTechnologies = new List<SasCellTechnology> { SasCellTechnology.Csi, SasCellTechnology.ThinFilm, SasCellTechnology.UserDefined };
      SolarArrayCellTechnologySelection = SasCellTechnology.Csi;
    }

    private void UpdateFunctionGeneratorValuesFromDevice()
    {
      if (_connectionService.IsConnected)
      {
        var functionGeneratorValues = _functionGeneratorService.ReadFunctionGeneratorValuesFromDevice();
        FunctionGeneratorControlModeSelection = functionGeneratorValues.ControlMode;
        FunctionGeneratorFunctionTypeSelection = functionGeneratorValues.FunctionType;
        BlockFunctionRepetitionsString = functionGeneratorValues.FunctionRepetitions.ToString();
        BlockPeakValueString = functionGeneratorValues.PeakValue.ToString(_cultureInfo);
        BlockOffsetString = functionGeneratorValues.Offset.ToString(_cultureInfo);
        BlockFrequencyString = functionGeneratorValues.Frequency.ToString(_cultureInfo);
        BlockSymmetryString = functionGeneratorValues.Symmetry.ToString(_cultureInfo);

        FunctionGeneratorTriggerModeSelection = functionGeneratorValues.TriggerMode;
        FunctionGeneratorSequenceFinishBehaviourSelection = functionGeneratorValues.FinishBehaviour;
        BlockRepetitionsString = functionGeneratorValues.BlockRepetitions.ToString();
        SequenceDelayString = functionGeneratorValues.SequenceDelay.ToString(_cultureInfo);
      }
    }

    private void UpdateSasSpecificFunctionGeneratorValuesFromDevice()
    {
      if(_connectionService.IsConnected)
      {
        var sasSpecificFunctionGeneratorValues = _functionGeneratorService.ReadSasSpecificFunctionGeneratorValuesFromDevice();
        MppVoltageString = sasSpecificFunctionGeneratorValues.NominalMppVoltage.ToString(_cultureInfo);
        MppCurrentString = sasSpecificFunctionGeneratorValues.NominalMppCurrent.ToString(_cultureInfo);
        ActualIrradianceString = sasSpecificFunctionGeneratorValues.ActualIrradiance.ToString(_cultureInfo);
        ActualTemperatureString = sasSpecificFunctionGeneratorValues.ActualTemperature.ToString(_cultureInfo);
      }
    }

    private void SetSasCurve(float mppVoltage, float mppCurrent, SasCellTechnology cellTechnology)
    {
      _functionGeneratorService.SetSasCurve(mppVoltage, mppCurrent, cellTechnology, CellTechnologyParametersViewModel.GetSolarArraySimulationCellTechnologyParameters());
      UpdateSasSpecificFunctionGeneratorValuesFromDevice();
    }

    private void SetFunctionGeneratorCommand(FunctionGeneratorCommand functionGeneratorCommand)
    {
      _functionGeneratorService.SetFunctionGeneratorCommand(functionGeneratorCommand);
      ReadFunctionGeneratorState();
    }

    private void ReadFunctionGeneratorState()
    {
      if (_connectionService.IsConnected)
      {
        FunctionGeneratorState actualState = _functionGeneratorService.GetFunctionGeneratorState();
        FunctionGeneratorIsEnabled = actualState != FunctionGeneratorState.Disabled;
        FunctionGeneratorIsRunning = actualState == FunctionGeneratorState.Paused || actualState == FunctionGeneratorState.Running;
      }
      else
      {
        FunctionGeneratorIsEnabled = false;
        FunctionGeneratorIsRunning = false;
      }
    }

    private void UpdateCellParameters()
    {
      float ffv;
      float ffi;
      float alpha;
      float beta;
      float cg;
      float cv;
      float cr;
      float nu2Lh;

      switch (SolarArrayCellTechnologySelection)
      {
        case SasCellTechnology.Csi:
          ffv = 0.8f;
          ffi = 0.9f;
          alpha = 0.0004f;
          beta = -0.004f;
          cg = 0.002514f;
          cv = 0.08593f;
          cr = 0.000109f;
          nu2Lh = 0.95f;
          break;
        case SasCellTechnology.ThinFilm:
          ffv = 0.72f;
          ffi = 0.8f;
          alpha = 0.0002f;
          beta = -0.002f;
          cg = 0.001252f;
          cv = 0.08419f;
          cr = 0.000148f;
          nu2Lh = 0.98f;
          break;
        case SasCellTechnology.UserDefined:
          //Just leave them on the old values;
          ffv = CellTechnologyParametersViewModel.Ffv;
          ffi = CellTechnologyParametersViewModel.Ffi;
          alpha = CellTechnologyParametersViewModel.Alpha;
          beta = CellTechnologyParametersViewModel.Beta;
          cg = CellTechnologyParametersViewModel.Cg;
          cv = CellTechnologyParametersViewModel.Cv;
          cr = CellTechnologyParametersViewModel.Cr;
          nu2Lh = CellTechnologyParametersViewModel.Nu2Lh;
          break;
        default:
          throw new NotImplementedException();
      }

      CellTechnologyParametersViewModel.AlphaString = alpha.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.BetaString = beta.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.FfvString = ffv.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.FfiString = ffi.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.CgString = cg.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.CvString = cv.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.CrString = cr.ToString(_cultureInfo);
      CellTechnologyParametersViewModel.NuL2HString = nu2Lh.ToString(_cultureInfo);
    }
  }
}
