﻿using Regatron.G5.Api.System;
using Regatron.G5.Api;
using System;
using Regatron.G5.Common;

namespace G5.Api.Example.Model
{
  internal class FunctionGeneratorService : BaseService, IFunctionGeneratorService
  {
    private readonly G5System _system;

    public FunctionGeneratorService(G5System system) 
    {
      _system = system ?? throw new ArgumentNullException();
    }

    public FunctionGeneratorValuesDTO ReadFunctionGeneratorValuesFromDevice()
    {
      FunctionGeneratorValuesDTO functionGeneratorValuesDTO = default;
      try
      {
        functionGeneratorValuesDTO = new FunctionGeneratorValuesDTO(
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetControlMode(),
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetBaseFunction(),
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetBaseFunctionRepetitions(),
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetPeakValue(),
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetOffset(),
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetFrequency(),
          _system.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration().GetSymmetry(),
          _system.GetFunctionGenerator().GetFunctionGeneratorSequenceConfiguration().GetTriggerMode(),
          _system.GetFunctionGenerator().GetFunctionGeneratorSequenceConfiguration().GetSequenceFinishBehaviour(),
          _system.GetFunctionGenerator().GetFunctionGeneratorSequenceConfiguration().GetBlockRepetitions(),
          _system.GetFunctionGenerator().GetFunctionGeneratorSequenceConfiguration().GetDelayBetweenBlockRepetitions()
          );
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return functionGeneratorValuesDTO;
    }

    public void SetFunctionGeneratorCommand(FunctionGeneratorCommand functionGeneratorCommand)
    {
      try
      {
        _system.GetFunctionGenerator().SetCommand(functionGeneratorCommand);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public FunctionGeneratorState GetFunctionGeneratorState()
    {
      FunctionGeneratorState functionGeneratorState = FunctionGeneratorState.Disabled;
      try
      {
        functionGeneratorState = _system.GetFunctionGenerator().GetState();
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return functionGeneratorState;
    }

    public void SetControlMode(FunctionGeneratorControlMode controlMode)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetControlMode(controlMode);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetBaseFunction(FunctionGeneratorFunctionType functionType)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetBaseFunction(functionType);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetBlockFunctionRepetitions(uint blockFunctionRepetitions)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetBaseFunctionRepetitions(blockFunctionRepetitions);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetPeakValue(float peakValue)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetPeakValue(peakValue);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetOffset(float offset)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetOffset(offset);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetFrequency(float frequency)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetFrequency(frequency);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetSymmetry(float symmetry)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorBlockConfiguration().SetSymmetry(symmetry);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetTriggerMode(FunctionGeneratorTriggerMode triggerMode)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorSequenceConfiguration().SetTriggerMode(triggerMode);
      }
      catch (G5ApiException g5ApiException)
      { 
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetSequenceFinishBehavior(FunctionGeneratorSequenceFinishBehaviour sequenceFinishBehavior)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorSequenceConfiguration().SetSequenceFinishBehaviour(sequenceFinishBehavior);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetBlockRepetitions(uint blockRepetitions)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorSequenceConfiguration().SetBlockRepetitions(blockRepetitions);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetSequenceDelay(float sequenceDelay)
    {
      try
      {
        _system.GetFunctionGenerator()?.GetFunctionGeneratorSequenceConfiguration().SetDelayBetweenBlockRepetitions(sequenceDelay);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetSasCurve(float mppVoltage, float mppCurrent, SasCellTechnology sasCellTechnology, 
      SolarArraySimulationCellTechnologyParameters cellTechnologyParameters)
    {
      switch (sasCellTechnology)
      {
        case SasCellTechnology.Csi:
          _system.GetSolarArraySimulation()?.SetCsiCurveWithMppData(mppVoltage, mppCurrent);
          break;
        case SasCellTechnology.ThinFilm:
          _system.GetSolarArraySimulation()?.SetThinFilmCurveWithMppData(mppVoltage, mppCurrent);
          break;
        case SasCellTechnology.UserDefined:
          _system.GetSolarArraySimulation()?.SetUserDefinedTechnologyCurveWithMppData(cellTechnologyParameters, mppVoltage, mppCurrent);
          break;
        default:
          throw new NotImplementedException();
      }
    }

    public SasSpecificFunctionGeneratorValuesDTO ReadSasSpecificFunctionGeneratorValuesFromDevice()
    {
      SasSpecificFunctionGeneratorValuesDTO sasSpecificFunctionGeneratorValuesDTO = default;
      try
      {
        sasSpecificFunctionGeneratorValuesDTO = new SasSpecificFunctionGeneratorValuesDTO(
          _system.GetSolarArraySimulation().GetCalculatedMppVoltage(),
          _system.GetSolarArraySimulation().GetCalculatedMppCurrent(),
          _system.GetSolarArraySimulation().GetActualIrradiance(),
          _system.GetSolarArraySimulation().GetActualTemperature());
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return sasSpecificFunctionGeneratorValuesDTO;
    }

    public void SetActualIrradiance(float actualIrradiance, float rampTime)
    {
      try
      {
        _system.GetSolarArraySimulation()?.SetActualIrradiance(actualIrradiance, rampTime);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetActualTemperature(float actualTemperature, float rampTime)
    {
      try
      {
        _system.GetSolarArraySimulation()?.SetActualTemperature(actualTemperature, rampTime);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }
  }
}
