﻿using Regatron.G5.Api.System;
using Regatron.G5.Api;
using Regatron.G5.Common;
using System;
using System.Text;

namespace G5.Api.Example.Model
{
  internal class DeviceInformationService : BaseService, IDeviceInformationService
  {
    private readonly G5System _system;

    public DeviceInformationService(G5System system)
    {
      _system = system ?? throw new ArgumentNullException();
    }

    public void SetVoltageControlMode()
    {
      try
      {
        _system.GetReferenceValues().SetControllerMode(ControllerMode.VoltageControlled);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public ReferenceValueDTO ReadReferenceValuesFromDevice()
    {
      ReferenceValueDTO referenceValueDTO = default;
      try
      {
        referenceValueDTO = new ReferenceValueDTO(
          _system.GetReferenceValues().GetVoltage(),
          _system.GetReferenceValues().GetMaximumCurrent(),
          _system.GetReferenceValues().GetMaximumPower());
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return referenceValueDTO;
    }

    public string GetActualErrorsAsString()
    {
      string actualError = string.Empty;
      try
      {
        if (_system.GetConnectedDevice().GetState().HasErrors())
        {
          StringBuilder stringBuilder = new StringBuilder();
          foreach (Incident incident in _system.GetConnectedDevice().GetState().GetIncidents())
          {
            stringBuilder.AppendLine($"code: {incident.Code} // group: {incident.Group} // type: {incident.Type} // time stamp: {incident.TimeStamp}");
          }
          actualError = stringBuilder.ToString();
        }
        else
        {
          actualError = "Device has no errors.";
        }
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return actualError;
    }

    public string GetDeviceStateAsString()
    {
      string devicesState = string.Empty;
      try
      {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.AppendLine($"Serial number: {_system.GetConnectedDevice().GetInformation().GetSerialNumber()}");
        stringBuilder.AppendLine($"Device name: {_system.GetConnectedDevice().GetInformation().GetDeviceName()}");
        stringBuilder.AppendLine($"Device state: {_system.GetConnectedDevice().GetState().GetState()}");
        stringBuilder.AppendLine($"Device has errors: {_system.GetConnectedDevice().GetState().HasErrors()}");
        stringBuilder.AppendLine($"Device has warnings: {_system.GetConnectedDevice().GetState().HasWarnings()}");
        devicesState = stringBuilder.ToString();
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return devicesState;
    }

    public bool IsVoltageOn()
    {
      bool isVoltageOn = default;
      try
      {
        isVoltageOn = _system.GetState().IsVoltageOn();
      }
      catch(G5ApiException g5ApiException) 
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }

      return isVoltageOn;
    }

    public void SetVoltageOn()
    {
      try
      {
        _system.GetCommands().SetVoltageOn();
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetVoltageOff()
    {
      try
      {
        _system.GetCommands().SetVoltageOff();
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetReferenceVoltage(float referenceVoltage)
    {
      try
      {
        _system.GetReferenceValues().SetVoltage(referenceVoltage);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetMaximumCurrent(float maximumCurrent)
    {
      try
      {
        _system.GetReferenceValues().SetMaximumCurrent(maximumCurrent);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }

    public void SetMaximumPower(float maximumPower)
    {
      try
      {
        _system.GetReferenceValues().SetMaximumPower(maximumPower);
      }
      catch (G5ApiException g5ApiException)
      {
        NotifyErrorMessageChanged(g5ApiException.Message);
      }
    }
  }
}
