import os
runtimeConfigFilePath = "Regatron.G5.Api.runtimeconfig.json" #The path to the runtime configuration file

from clr_loader import get_coreclr
from pythonnet import set_runtime
runtime = get_coreclr(runtime_config=runtimeConfigFilePath)
set_runtime(runtime)

import clr #Imports the pythonnet to use .Net functionality in python.

clr.AddReference('Regatron.G5.Api') #Adds the Regatron.G5.Api assembly. The Regatron.G5.Api.dll and Regatron.G5.FunctionGeneratorNative.dll have to be placed right next to this script.
from Regatron.G5.Api import G5ApiException
from Regatron.G5.Api.System import G5System
from Regatron.G5.Common import FunctionGeneratorTriggerMode
from Regatron.G5.Common import FunctionGeneratorFunctionType
from Regatron.G5.Common import FunctionGeneratorControlMode
from Regatron.G5.Common import FunctionGeneratorSequenceFinishBehaviour
from Regatron.G5.Common import FunctionGeneratorCommand
from Regatron.G5.Common import ControllerMode

#Create the number of G5System instances you need
systemList = [G5System.CreateSystem(), G5System.CreateSystem()]
systemPromptString = "Choose system number (1 - " + str(len(systemList)) + "): "

#Connect to a device if not yet a connection is established.
def connect(connect: bool, ip: str, port: int, systemInstance):
    if connect:
        if not systemInstance.IsConnected():
            print(systemInstance.Connect(ip, port))
        else:
            print("A device is already connected.")
    else:
        print(systemInstance.Disconnect())

#Gets the actual values for voltage, current and power.
def getActualValues(systemInstance):
    if systemInstance.IsConnected():
        actualSystemValuesInstance = systemInstance.GetActualValues()
        print("voltage: " + str(actualSystemValuesInstance.GetOutputVoltage()) + "\ncurrent: " + str(actualSystemValuesInstance.GetOutputCurrent()) + "\npower: " + str(actualSystemValuesInstance.GetOutputPower()))
    else:
        print("No device connected.")
        
#Sets the reference values for voltage, the upper limit for current and power and sets the device to voltage controlled.
def setReferenceValues(voltageReference, currentUpperLimit, powerUpperLimit, systemInstance):
    if systemInstance.IsConnected():
        result = True
        errorMessage = ""
        try:
            referenceValuesInstance = systemInstance.GetReferenceValues()
            referenceValuesInstance.SetControllerMode(ControllerMode.VoltageControlled)
            referenceValuesInstance.SetVoltage(float(voltageReference))
            referenceValuesInstance.SetMaximumCurrent(float(currentUpperLimit))
            referenceValuesInstance.SetMaximumPower(float(powerUpperLimit))
            print("voltage: " + str(referenceValuesInstance.GetVoltage()) + "\ncurrent: " + str(referenceValuesInstance.GetMaximumCurrent()) + "\npower: " + str(referenceValuesInstance.GetMaximumPower()))
        except G5ApiException as exception:
            result = False
            errorMessage = exception.Message
    else:
        errorMessage = "No device connected."
        result = False
    return result, errorMessage
        
#Sets the function generator block.
def setFunctionGeneratorBlock(physicalPreset, function, peakValue, offset, frequency, symmetry, functionRepetitions, systemInstance):
    if systemInstance.IsConnected():
        result = True
        errorMessage = ""
        try:
            functionGeneratorBlockConfigurationInstance = systemInstance.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration()
            functionGeneratorBlockConfigurationInstance.SetControlMode(physicalPreset)
            functionGeneratorBlockConfigurationInstance.SetBaseFunction(function)
            functionGeneratorBlockConfigurationInstance.SetPeakValue(float(peakValue))
            functionGeneratorBlockConfigurationInstance.SetOffset(float(offset))
            functionGeneratorBlockConfigurationInstance.SetFrequency(float(frequency))
            functionGeneratorBlockConfigurationInstance.SetSymmetry(float(symmetry))
            functionGeneratorBlockConfigurationInstance.SetBaseFunctionRepetitions(functionRepetitions)
        except G5ApiException as exception:
            result = False
            errorMessage = exception.Message
    else:  
        result = False
        errorMessage = "No device connected."
    return result, errorMessage
        
#Sets the function generator sequence.
def setFunctionGeneratorSequence(triggerMode, behaviourWhenFinished, blockRepetitions, delay, systemInstance):
    if systemInstance.IsConnected():
        result = True
        errorMessage = ""
        try:
            functionGeneratorSequenceConfigurationInstance = systemInstance.GetFunctionGenerator().GetFunctionGeneratorSequenceConfiguration()
            functionGeneratorSequenceConfigurationInstance.SetTriggerMode(triggerMode)
            functionGeneratorSequenceConfigurationInstance.SetSequenceFinishBehaviour(behaviourWhenFinished)
            functionGeneratorSequenceConfigurationInstance.SetBlockRepetitions(blockRepetitions)
            functionGeneratorSequenceConfigurationInstance.SetDelayBetweenBlockRepetitions(float(delay))
        except G5ApiException as exception:
            result = False
            errorMessage = exception.Message
    else:
        result = False
        errorMessage = "No device connected."
    return result, errorMessage

#Gets some system information.
def getSystemState(systemInstance):
    if systemInstance.IsConnected():
        print("SN: " + str(systemInstance.GetConnectedDevice().GetInformation().GetSerialNumber()))
        print("Device name: " + systemInstance.GetConnectedDevice().GetInformation().GetDeviceName())
        print("Device state: " + str(systemInstance.GetConnectedDevice().GetState().GetState()))
        print("Device has errors: " + str(systemInstance.GetConnectedDevice().GetState().HasErrors()))
        print("Device has warnings: " + str(systemInstance.GetConnectedDevice().GetState().HasWarnings()))
    else:  
        print("No device connected.")
        
#Gets the actual incidents.
def getActualIncidents(systemInstance):
    if systemInstance.IsConnected():
        print("Device has errors: " + str(systemInstance.GetConnectedDevice().GetState().HasErrors()))
        listOfIncidents = systemInstance.GetConnectedDevice().GetState().GetIncidents()
        for incident in listOfIncidents:
            print("Incident code: " + str(incident.Code))
            print("Incident group: " + str(incident.Group))
            print("Incident type: " + str(incident.Type))
            print("Incident time stamp: " + str(incident.TimeStamp))
    else:  
        print("No device connected.")

#Set voltage on.        
def setVoltageOn(systemInstance):
    if systemInstance.IsConnected():
        try:
            result = True
            errorMessage = ""
            systemInstance.GetCommands().SetVoltageOn()
        except G5ApiException as exception:
            result = False
            errorMessage = exception.Message
    else:  
        result = False
        errorMessage = "No device connected."
    return result, errorMessage
        
#Set voltage off. 
def setVoltageOff(systemInstance):
    if systemInstance.IsConnected():
        try:
            result = True
            errorMessage = ""
            systemInstance.GetCommands().SetVoltageOff()
        except G5ApiException as exception:
            result = False
            errorMessage = exception.Message
    else:  
        result = False
        errorMessage = "No device connected."
    return result, errorMessage
        
#Invokes the function generator commands.
def setFunctionGeneratorCommand(functionGeneratorCommand, systemInstance):
    if systemInstance.IsConnected():
        try:
            result = True
            errorMessage = ""
            systemInstance.GetFunctionGenerator().SetCommand(functionGeneratorCommand)
        except G5ApiException as exception:
            result = False
            errorMessage = exception.Message
    else:  
        result = False
        errorMessage = "No device connected."
    return result, errorMessage
                      
#Prints a dotted line to the console window.
def printDotedLine():
    print("--------------------------------------------------")

#"User-Interface" to ask the user what to do.
def askinput():
    result = True 
    printDotedLine()
    choice = input("Please enter a number to start an action\n1: connect to device\n2: disconnect from device\n3: set voltage control \n4: get actual values \n5: initialize function generator \n6: set voltage on \n7: set voltage off \n8: enable function generator \n9: disable function generator  \n10: start function generator \n11: stop function generator \n12: get system state \n13: get actual incidents \nelse: exit\ninput: ")
    printDotedLine()
    if choice == "1":
        systemNumber = int(input(systemPromptString))
        ip = input("ip: ")
        port = input("port: ")
        connect(True, ip, int(port), systemList[systemNumber - 1])
    elif choice == "2":
        systemNumber = int(input(systemPromptString))
        connect(False, 0, 0, systemList[systemNumber - 1])
    elif choice == "3":
        systemNumber = int(input(systemPromptString))
        voltageReference = input("voltage reference [V]: ")
        currenUppertLimit = input("current upper limit [A]: ")
        powerUpperLimit = input("power upper limit [W]: ")
        (setResult, errorMessage) = setReferenceValues(voltageReference, currenUppertLimit, powerUpperLimit, systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "4":
        systemNumber = int(input(systemPromptString))
        getActualValues(systemList[systemNumber - 1])
    elif choice == "5":
        systemNumber = int(input(systemPromptString))
        (resultBlock, errorMessageBlock) = setFunctionGeneratorBlock(FunctionGeneratorControlMode.Voltage, FunctionGeneratorFunctionType.Sine, 213.2, 13.2, 51.5, 0.5, 0, systemList[systemNumber - 1])
        (resultSequence, errorMessageSequence) = setFunctionGeneratorSequence(FunctionGeneratorTriggerMode.Manual, FunctionGeneratorSequenceFinishBehaviour.VoltageOff, 1, 0, systemList[systemNumber - 1])
        if not resultBlock:
            print(errorMessageBlock)
        if not resultSequence:
            print(errorMessageSequence)
    elif choice == "6":
        systemNumber = int(input(systemPromptString))
        (setResult, errorMessage) = setVoltageOn(systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "7":
        systemNumber = int(input(systemPromptString))
        (setResult, errorMessage) = setVoltageOff(systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "8":
        systemNumber = int(input(systemPromptString))
        (setResult, errorMessage) = setFunctionGeneratorCommand(FunctionGeneratorCommand.Enable, systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "9":
        systemNumber = int(input(systemPromptString))
        (setResult, errorMessage) = setFunctionGeneratorCommand(FunctionGeneratorCommand.Disable, systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "10":
        systemNumber = int(input(systemPromptString))
        (setResult, errorMessage) = setFunctionGeneratorCommand(FunctionGeneratorCommand.ManualStart, systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "11":
        systemNumber = int(input(systemPromptString))
        (setResult, errorMessage) = setFunctionGeneratorCommand(FunctionGeneratorCommand.ManualStop, systemList[systemNumber - 1])
        if not setResult:
            print(errorMessage)
    elif choice == "12":
        systemNumber = int(input(systemPromptString))
        getSystemState(systemList[systemNumber - 1])
    elif choice == "13":
        systemNumber = int(input(systemPromptString))
        getActualIncidents(systemList[systemNumber - 1])
    else:
        result = False
    return result

#Endless loop until the user stops it.
while askinput():
    pass

#Make sure all systems are disconnected
for index in range(0, len(systemList)):
    systemList[index].Disconnect()
