﻿#Set-ExecutionPolicy -ExecutionPolicy Unrestricted -Scope Process
using namespace System.Collections.Generic

#load dll
$FullPathToG5Api = $PSScriptRoot + "\Regatron.G5.Api.dll" #Place the Regatron.G5.Api.dll right next to this script
try
{       
    Add-Type -Path $FullPathToG5Api
    Write-Host "Assembly loaded:" $FullPathToG5Api
}
catch
{
    Write-Host "Assembly not loaded:" $FullPathToG5Api
}

Write-Host ""

#Create the count of G5System instances you need
$g5SystemInstance1 = [Regatron.G5.Api.System.G5System]::CreateSystem()
$g5SystemInstance2 = [Regatron.G5.Api.System.G5System]::CreateSystem()
$systemList = New-Object System.Collections.Generic.List[Regatron.G5.Api.System.G5System]
$systemList.Add($g5SystemInstance1);
$systemList.Add($g5SystemInstance2);
$systemPromptString = "Choose system number (1 - " + $systemList.Count + ")" 


function printDotedLine()
{
    Write-Host "--------------------------------------------------"
}

#Connect to a device if not yet a connection is established.
function connect($connect, [string]$ip, [uint16]$port, $systemInstance)
{
    if($connect)
    {
        if(-Not $systemInstance.IsConnected())
        {
            Write-Host "ip" + $ip
            Write-Host "port" + $port
            Write-Host $systemInstance.Connect($ip, $port)
        }
        else
        {
            Write-Host "A device is already connected."
        }
    }
    else
    {
        Write-Host $systemInstance.Disconnect()
    }
}

#Gets the actual values for voltage, current and power.
function getActualValues($systemInstance)
{
    if($systemInstance.IsConnected())
    {
        $actualSystemValuesInstance = $systemInstance.GetActualValues()
        Write-Host "voltage:" $actualSystemValuesInstance.GetOutputVoltage()
        Write-Host "current:" $actualSystemValuesInstance.GetOutputCurrent()
        Write-Host "power:" $actualSystemValuesInstance.GetOutputPower()
    }
    else
    {
        Write-Host "No device connected."
    }
}

#Sets the function generator block.
function setFunctionGeneratorBlock([Regatron.G5.Common.FunctionGeneratorControlMode]$controlMode, [Regatron.G5.Common.FunctionGeneratorFunctionType]$function, [float]$peakValue, [float]$offset, [float]$frequency, [float]$symmetry, [uint32]$functionRepetitions, $systemInstance)
{
    $result = $false
    $errorMessage = ''
    if($systemInstance.IsConnected())
    {
        try
        {
            $functionGeneratorBlockConfigurationInstance = $systemInstance.GetFunctionGenerator().GetFunctionGeneratorBlockConfiguration()
            $functionGeneratorBlockConfigurationInstance.SetControlMode($controlMode)
            $functionGeneratorBlockConfigurationInstance.SetBaseFunction($function)
            $functionGeneratorBlockConfigurationInstance.SetPeakValue($peakValue)
            $functionGeneratorBlockConfigurationInstance.SetOffset($offset)
            $functionGeneratorBlockConfigurationInstance.SetFrequency($frequency)
            $functionGeneratorBlockConfigurationInstance.SetSymmetry($symmetry)
            $functionGeneratorBlockConfigurationInstance.SetBaseFunctionRepetitions($functionRepetitions)
            $result = $true
        }
        catch [Regatron.G5.Api.G5ApiException]
        {
            $errorMessage = $PSItem.Exception.Message
        }

    }
    else
    {
        $errorMessage = "No device connected."
    }
    return $result, $errorMessage
}

#Sets the function generator sequence.
function setFunctionGeneratorSequence([Regatron.G5.Common.FunctionGeneratorTriggerMode]$triggerMode, [Regatron.G5.Common.FunctionGeneratorSequenceFinishBehaviour]$behaviourWhenFinished, [uint32]$blockRepetitions, [float]$delay, $systemInstance)
{
    $result = $false
    $errorMessage = ''
    if($systemInstance.IsConnected())
    {
        try
        {
            $functionGeneratorSequenceConfigurationInstance = $systemInstance.GetFunctionGenerator().GetFunctionGeneratorSequenceConfiguration()
            $functionGeneratorSequenceConfigurationInstance.SetTriggerMode($triggerMode)
            $functionGeneratorSequenceConfigurationInstance.SetSequenceFinishBehaviour($behaviourWhenFinished)
            $functionGeneratorSequenceConfigurationInstance.SetBlockRepetitions($blockRepetitions)
            $functionGeneratorSequenceConfigurationInstance.SetDelayBetweenBlockRepetitions($delay)
            $result = $true
        }
        catch [Regatron.G5.Api.G5ApiException]
        {
            $errorMessage = $PSItem.Exception.Message
        }

    }
    else
    {
        $errorMessage = "No device connected."
    }
    return $result, $errorMessage
}

#Sets the reference values for voltage, the upper limit for current and power and sets the device to voltage controlled.
function setReferenceValues([float]$voltageReference, [float]$currentUpperLimit, [float]$powerUpperLimit, $systemInstance)
{
    $result = $false
    $errorMessage = ''
    if($systemInstance.IsConnected())
    {
        try
        {
            $referenceValuesInstance = $systemInstance.GetReferenceValues()
            $referenceValuesInstance.SetControllerMode([Regatron.G5.Common.ControllerMode]::VoltageControlled)
            $referenceValuesInstance.SetVoltage($voltageReference)
            $referenceValuesInstance.SetMaximumCurrent($currentUpperLimit)
            $referenceValuesInstance.SetMaximumPower($powerUpperLimit)
            printDotedLine
            Write-Host "voltage:" $referenceValuesInstance.GetVoltage()
            Write-Host "current:" $referenceValuesInstance.GetMaximumCurrent()
            Write-Host "power:" $referenceValuesInstance.GetMaximumPower()
        }
        catch [Regatron.G5.Api.G5ApiException]
        {
            $errorMessage = $PSItem.Exception.Message
        }

    }
    else
    {
        $errorMessage = "No device connected."
    }
    return $result, $errorMessage
}

#Set voltage on.
function setVoltageOn($systemInstance)
{
    $result = $false
    $errorMessage = ''
    if($systemInstance.IsConnected())
    {
        try
        {
            $systemInstance.GetCommands().SetVoltageOn()
            $result = $true
        }
        catch [Regatron.G5.Api.G5ApiException]
        {
            $errorMessage = $PSItem.Exception.Message
        }

    }
    else
    {
        $errorMessage = "No device connected."
    }
    return $result, $errorMessage
}

#Set voltage on.
function setVoltageOff($systemInstance)
{
    $result = $false
    $errorMessage = ''
    if($systemInstance.IsConnected())
    {
        try
        {
            $systemInstance.GetCommands().SetVoltageOff()
            $result = $true
        }
        catch [Regatron.G5.Api.G5ApiException]
        {
            $errorMessage = $PSItem.Exception.Message
        }

    }
    else
    {
        $errorMessage = "No device connected."
    }
    return $result, $errorMessage
}

#Invokes the function generator commands.
function setFunctionGeneratorCommand([Regatron.G5.Common.FunctionGeneratorCommand]$functionGeneratorCommand, $systemInstance)
{
    $result = $false
    $errorMessage = ''
    if($systemInstance.IsConnected())
    {
        try
        {
            $systemInstance.GetFunctionGenerator().SetCommand($functionGeneratorCommand)
            $result = $true
        }
        catch [Regatron.G5.Api.G5ApiException]
        {
            $errorMessage = $PSItem.Exception.Message
        }

    }
    else
    {
        $errorMessage = "No device connected."
    }
    return $result, $errorMessage
}

#Gets some system information.
function getSystemState($systemInstance)
{
    printDotedLine
    if($systemInstance.IsConnected())
    {
        Write-Host "SN:" $systemInstance.GetConnectedDevice().GetInformation().GetSerialNumber()
        Write-Host "Device name:" $systemInstance.GetConnectedDevice().GetInformation().GetDeviceName()
        Write-Host "Device state:" $systemInstance.GetConnectedDevice().GetState().GetState()
        Write-Host "Device has errors:" $systemInstance.GetConnectedDevice().GetState().HasErrors()
        Write-Host "Device has warnings:" $systemInstance.GetConnectedDevice().GetState().HasWarnings()
    }
    else
    {
        Write-Host "No device connected."
    }
}   

#Gets the actual system incidents.
function getActualIncidents($systemInstance)
{
    printDotedLine
    if($systemInstance.IsConnected())
    {
        Write-Host "Device has errors:" $systemInstance.GetConnectedDevice().GetState().HasErrors()
        $actualIncidents = $systemInstance.GetConnectedDevice().GetState().GetIncidents()
        foreach($incident in $actualIncidents)
        {
            printDotedLine
            Write-Host "Incident code:" $incident.Code
            Write-Host "Incident group:" $incident.Group
            Write-Host "Incident type:" $incident.Type
            Write-Host "Incident time stamp:" $incident.TimeStamp.ToString()
        }
    }
    else
    {
        Write-Host "No device connected."
    }
}    

#"User-Interface" to ask the user what to do.
try
{
    $runProgram = $true
    while($runProgram)
    {
        printDotedLine
        [int]$inp = Read-Host -Prompt "Please enter a number to start an action `n1: connect to device`n2: disconnect from device`n3: set voltage control  `n4: get actual values `n5: initialize function generator `n6: set voltage on `n7: set voltage off `n8: enable function generator `n9: disable function generator `n10: start function generator `n11: stop function generator `n12: get system state `n13: get actual incidents `nelse: exit`ninput"
        $runProgram = $inp -lt 14 -and $inp -ge 0
        printDotedLine
        switch($inp)
        {
            1 
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $ip = Read-Host -Prompt "IP"
                $port = Read-Host -Prompt "Port"
                connect $true $ip $port $systemList[$systemNumber - 1]
            }
            2
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                connect $false "0" 0 $systemList[$systemNumber - 1]
            }
            3 
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $voltageReference = Read-Host -Prompt "voltage reference"
                $currenUppertLimit = Read-Host -Prompt "current upper limit"
                $powerUpperLimit = Read-Host -Prompt "power upper limit"
                $powerUpperLimit, $errorMessage = setReferenceValues $voltageReference $currenUppertLimit $powerUpperLimit $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                } 
            }
            4 
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                getActualValues $systemList[$systemNumber - 1]
            }
            5
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                ($resultBlock, $errorMessageBlock) = setFunctionGeneratorBlock ([Regatron.G5.Common.FunctionGeneratorControlMode]::Voltage) ([Regatron.G5.Common.FunctionGeneratorFunctionType]::Sine) 213.2 13.2 51.5 0.5 0 $systemList[$systemNumber - 1]
                ($resultSequence, $errorMessageSequence) = setFunctionGeneratorSequence ([Regatron.G5.Common.FunctionGeneratorTriggerMode]::Manual) ([Regatron.G5.Common.FunctionGeneratorSequenceFinishBehaviour]::VoltageOff) 1 0 $systemList[$systemNumber - 1]
                if (-Not $resultBlock)
                {
                    Write-Host $errorMessageBlock
                }
                if (-Not $resultSequence)
                {
                    Write-Host $errorMessageSequence
                }           
            }
            6
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $setResult, $errorMessage = setVoltageOn $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                }
            }
            7
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $setResult, $errorMessage = setVoltageOff $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                }
            }
            8
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $setResult, $errorMessage = setFunctionGeneratorCommand ([Regatron.G5.Common.FunctionGeneratorCommand]::Enable) $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                }
            }
            9
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $setResult, $errorMessage = setFunctionGeneratorCommand ([Regatron.G5.Common.FunctionGeneratorCommand]::Disable) $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                }
            }
            10
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $setResult, $errorMessage = setFunctionGeneratorCommand ([Regatron.G5.Common.FunctionGeneratorCommand]::ManualStart) $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                }
            }
            11
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                $setResult, $errorMessage = setFunctionGeneratorCommand ([Regatron.G5.Common.FunctionGeneratorCommand]::ManualStop) $systemList[$systemNumber - 1]
                if(-Not $setResult)
                {
                    Write-Host $errorMessage
                }
            }
            12
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                getSystemState $systemList[$systemNumber - 1]
            }
            13
            {
                $systemNumber = Read-Host -Prompt $systemPromptString
                getActualIncidents $systemList[$systemNumber - 1]
            }
            default 
            {
                printDotedLine
            }
        }
    }
}
catch
{
    Write-Host $_.ScriptStackTrace
    Write-Host $_
    Write-Host $PSItem.Exception.StackTrace
    Write-Host $PSItem.Exception.InnerException
}
finally
{
    Write-Host "Program terminated"   
	# Disconnect
    foreach ($system in $systemList)
    {
        $system.Disconnect()
        $isConnected = $system.IsConnected()
        Write-Host "IsConnected:" $isConnected
    }
}